/**********************************************************************
*	Bastion Products Copyright 1998								*
*	All rights reserved										*
*	This source is freely distributable in an unmodified state.		*
*	Project:			Blithe Shared Library						*
*	Source:			BlitheLib.c								*
*	Last Modified:	5/23/98									*
*	Author:			Jennifer Weston							*
*	Description:		This project creates a shared library which	*
*					allows applications to have dynamic text in	*
*					different languages.						*
***********************************************************************/
#include "BlitheConst.h"
#include "BlitheLib.h"
#include <Resources.h>
#include <Path.h>
#include <FindDirectory.h>
#include <cstring>
#include <cstdio>
#include <cstdlib>
#include <Directory.h>
#include <Roster.h>
#include <Application.h>
#include <NodeInfo.h>

#define BLITHE_LIB	"blithe.so"

// Used internally by several functions
BResources* FindSharedLib(void);
bool get_notify(void);
char *GetHex(int32 inStringID);
BResources *FindNextLocalDictionary(int32 inLangID,int32 i);
BResources *FindCommonDictionary(int32 inLangID);
BResources *FindDictionaryInPath(BPath thePath,int32 inLangID,int32 i);
int32 GetFirstDictionary(void);
BResources *FindAppDictionary(void);
void ConvertEscapes(char* entry);

/**********************************************************************
*	Method:		blithe_init									*
*	Arguments:	<none>										*
*	Returns:		status_t										*
*	Description:	This call is provided in case the user only links	*
*				against the shared library instead of the static	*
*				library.										*
**********************************************************************/
status_t blithe_init(void)
{
	return B_OK;
}

/**********************************************************************
*	Method:		has_application_dictionary						*
*	Arguments:	int32	inLangID								*
*							the requested langauge ID			*
*	Returns:		int		1 if the inLangID has an application		*
*						dictionary; 0 if not					*
*	Description:	Checks to make sure that the current language is 	*
*				has an application dictionary 					*
**********************************************************************/
int has_application_dictionary(int32 inLangID)
{
	BResources *local = FindNextLocalDictionary(inLangID,0);
	int hasAppDictionary = (local != NULL) ? 1 : 0;
	delete local;
	return hasAppDictionary;
}

/**********************************************************************
*	Method:		get_default_language							*
*	Arguments:	<none>										*
*	Returns:		int32	the language ID used when the 0 is		*
*						specified								*
*	Description:	Retrieves the language ID from the Blithe settings	*
*				file, which is set by the Blithe Settings app.		*
**********************************************************************/
int32 get_default_language(void)
{
	int32 *tmpID,currentID;
	BPath settingsPath;
	size_t idsize;
	status_t err;

/* Get the directory */
	err = find_directory(B_USER_SETTINGS_DIRECTORY,&settingsPath);
	if (err != B_NO_ERROR) return GetFirstDictionary();
	settingsPath.Append("blithe_settings");

/* Open the settings file as a resource for read only */
	BFile *settingsFile = new BFile(settingsPath.Path(),B_READ_ONLY);
	if (!settingsFile || settingsFile->InitCheck() != B_NO_ERROR) return GetFirstDictionary();
	BResources *settingsRes = new BResources(settingsFile,false);
	delete settingsFile;
	if (!settingsRes) return GetFirstDictionary();

/* Get the current language resource */
	tmpID = (int32*)(settingsRes->FindResource('LONG',1,&idsize));
	if (tmpID)
	{
		currentID = *tmpID;
		free(tmpID);
	} else currentID = GetFirstDictionary();

/* Clean up */
	delete settingsRes;
	return currentID;
}

/**********************************************************************
*	Method:		get_entry										*
*	Arguments:	int32	inStringID							*
*							the requested entry ID				*
*				int32	inLangID								*
*							the requested language ID			*
*				char*	inBuffer								*
*							optional buffer to store the string	*
*				int32	inBuffSize							*
*							length of inBuffer					*
*	Returns:		char*	the string containing the entry text		*
*	Description:	Retrieves the string specified by inStringID and	*
*				inLangID. If inBuffer is not null, inBuffSize-1 	*
*				characters are copied into it and it is returned.	*
**********************************************************************/
char* get_entry(int32 inStringID, int32 inLangID, char* inBuffer, int32 inBuffSize)
{
/***************************************
	Need to change this so it checks
	all dictionaries that match the
	specified criteria until it finds
	what it's looking for
***************************************/
/* Try the local dictionary */
	size_t len;
	char *entry=NULL;
	int32 i=0;
	BResources *theRes=NULL;
	while ((theRes = FindNextLocalDictionary(inLangID,i)) != NULL && !entry)
	{
		++i;
		if (theRes) entry = (char*)(theRes->FindResource('CSTR',inStringID,&len));
		delete theRes;
	}

/* If nothing, try the common dictionary */
	if (!entry)
	{
		delete theRes;
		theRes = FindCommonDictionary(inLangID);
		if (theRes) entry = (char*)(theRes->FindResource('CSTR',inStringID,&len));
		BResources *theCommonRes = theRes;

	/* If nothing still, try the application */
		if (!entry)
		{
			theRes = FindAppDictionary();
			if (theRes) entry = (char*)(theRes->FindResource('CSTR',inStringID,&len));

		/* As a last resort, make something up */
			if (!entry)
			{
				if (theCommonRes) entry = (char*)(theCommonRes->FindResource('CSTR',BL_MISSING_STRING,&len));
				if (!entry)
				{
					delete theCommonRes;
					return GetHex(inStringID);
				}
				if (strstr(entry,"%08x"))
				{
					char *theString = (char*)(malloc(strlen(entry)+11));
					sprintf(theString,entry,inStringID);
					free(entry);
					entry = theString;
				}
			}
		}
	}

/* Put away the file */
	delete theRes;

// Convert the escape codes to their proper form
	if (entry) ConvertEscapes(entry);

/* If not found, return error */
	if (!inBuffer || !inBuffSize) return entry;

/* Copy up to inBuffSize bytes into the inBuffer */
	strncpy(inBuffer,entry,inBuffSize-1);
	inBuffer[inBuffSize] = (char)0;
	free(entry);

	return inBuffer;
}

/**********************************************************************
*	Method:		get_blob											*
*	Arguments:	int32	inResType									*
*							the requested resource ID				*
*				int32	inResID										*
*							the requested resource type				*
*				int32	inLangID									*
*							the requested language ID				*
*				void*	inBuffer									*
*							a buffer for the data				*
*				size_t	inBuffLength									*
*							length of inBuffer				*
*				size_t*	outBlobSize									*
*							size in bytes of the data				*
*	Returns:		void*	the pointer to your data				*
*	Description:	Retrieves data from the dictionary by resource 	*
*				type and id, and inLangID. It requires a buffer  	*
*				for the output size (outBlobSize) which it assigns 	*
*				the number of bytes that are in the block of data	*
*				pointed to by the void pointer.						*
**********************************************************************/
void* get_blob(uint32 inResType,int32 inResID, int32 inLangID,void* inBuffer,size_t inBuffLength,size_t* outBlobSize)
{
/* Try the local dictionary */
	void *blob=NULL;
	int32 i=0;
	BResources *theRes=NULL;
	while ((theRes = FindNextLocalDictionary(inLangID,i)) != NULL && !blob)
	{
		++i;
		if (theRes) blob = theRes->FindResource((type_code)inResType,inResID,outBlobSize);
		delete theRes;
	}

/* If got it, return it */
	if (blob)
	{
		if (inBuffer == NULL || inBuffLength == 0)
		{
			inBuffLength = *outBlobSize;
			inBuffer = malloc(inBuffLength);
		}
		size_t t;
		t = (inBuffLength > *outBlobSize) ? inBuffLength : *outBlobSize; 
		memcpy(inBuffer,blob,t);
		delete theRes;
		return inBuffer;
	}

/* Try the common dictionary */
	theRes = FindCommonDictionary(inLangID);
	if (theRes) blob = theRes->FindResource((type_code)inResType,inResID,outBlobSize);

/* If got it, return it */
	if (blob)
	{
		if (inBuffer == NULL || inBuffLength == 0)
		{
			inBuffLength = *outBlobSize;
			inBuffer = malloc(inBuffLength);
		}
		size_t t;
		t = (inBuffLength > *outBlobSize) ? inBuffLength : *outBlobSize; 
		memcpy(inBuffer,blob,t);
		delete theRes;
		return inBuffer;
	}

/* Try the application; this is our last shot here */
	theRes = FindAppDictionary();
	if (theRes) blob = theRes->FindResource((type_code)inResType,inResID,outBlobSize);

/* If got it, return it */
	if (blob)
	{
		if (inBuffer == NULL || inBuffLength == 0)
		{
			inBuffLength = *outBlobSize;
			inBuffer = malloc(inBuffLength);
		}
		size_t t;
		t = (inBuffLength > *outBlobSize) ? inBuffLength : *outBlobSize; 
		memcpy(inBuffer,blob,t);
	}
	delete theRes;
	return inBuffer;
}

/**********************************************************************
*	Method:		get_entry_length								*
*	Arguments:	int32	inStringID							*
*							the requested entry ID				*
*				int32	inLangID								*
*							the requested language ID			*
*	Returns:		int32	Number of bytes						*
*	Description:	Retrieves the number of bytes of the string		*
*				specified by inStringID and inLangID.				*
**********************************************************************/
int32 get_entry_length(int32 inStringID, int32 inLangID)
{
/* Try the local dictionary */
	size_t len=0;
	char *entry=NULL;
	BResources *theRes;
	int32 i=0;
	while ((theRes = FindNextLocalDictionary(inLangID,i)) != NULL && !entry)
	{
		++i;
		if (theRes) entry = (char*)(theRes->FindResource('CSTR',inStringID,&len));
		delete theRes;
	}

/* If nothing, try the common dictionary */
	if (!entry)
	{
		delete theRes;
		theRes = FindCommonDictionary(inLangID);
		if (theRes) entry = (char*)(theRes->FindResource('CSTR',inStringID,&len));

	/* If nothing still, try the application */
		if (!entry)
		{
			theRes = FindAppDictionary();
			if (theRes) entry = (char*)(theRes->FindResource('CSTR',inStringID,&len));
		}
	}

/* Put away the file */
	delete theRes;

/* If not found, return error */
	if (!entry) return 0;
	free(entry);

/* Return the length */
	return len+1;
}

/**********************************************************************
*	Function:		get_max_length								*
*	Arguments:	int32	inLangID								*
*							the language ID that is requested	*
*	Returns:		int32	returns the maximum number of bytes for	*
*						inLangID								*
*	Description:	Checks the application, common, and built-in		*
*				dictionaries for max length						*
**********************************************************************/
int32 get_max_length(int32 inLangID)
{
	int32 appLength=0,commonLength=0,builtinLength=0,builtinAppLength=0,maxLength=0,*length=NULL;
	size_t	len;

// Check the application dictionary
	BResources *theRes;
	int32 i=0;
	while ((theRes = FindNextLocalDictionary(inLangID,i)) != NULL && !length)
	{
		if (theRes) length = (int32*)(theRes->FindResource('LONG',2,&len));
		delete theRes;
		++i;
	}

	if (length)
	{
		appLength = *length;
		free(length);
	}

// Check the common dictonary
	theRes = FindCommonDictionary(inLangID);
	if (theRes) length = (int32*)(theRes->FindResource('LONG',2,&len));
	delete theRes;
	if (length)
	{
		commonLength = *length;
		free(length);
	}

// Check the app dictionary
	theRes = FindAppDictionary();
	if (theRes) length = (int32*)(theRes->FindResource('LONG',2,&len));
	delete theRes;
	if (length)
	{
		builtinAppLength = *length;
		free(length);
	}

// Get the maximum length
	maxLength = appLength;
	if (commonLength > maxLength) maxLength = commonLength;
	if (builtinAppLength > maxLength) maxLength = builtinAppLength;
	if (builtinLength > maxLength) maxLength = builtinLength;
	return maxLength;
}

/**********************************************************************
*	Method:		get_language_list								*
*	Arguments:	char**	outList								*
*							buffer for the string array			*
*	Returns:		int32	Number of strings in the array			*
*	Description:	Builds an array of char*, each containing the name	*
*				of an installed language.						*
**********************************************************************/
int32 get_language_list(char** outList)
{
/* Get the common dictionary directory */
	status_t err;
	BPath thePath;
	err = find_directory(B_COMMON_ETC_DIRECTORY,&thePath);
	if (err != B_NO_ERROR) return 0;
	thePath.Append("Blithe");
	BDirectory theDirectory(thePath.Path());
	theDirectory.Rewind();
	int32 count = theDirectory.CountEntries();
	char **nameArray = (char**)(malloc(sizeof(char*)*count));
	BEntry candidate;
	size_t totalSize=0;
	int32 numDictionary=0;

/* Go through each dictionary found in the common dictionary dir */
	while (theDirectory.GetNextEntry(&candidate,true) == B_OK)
	{
	/* Get the name of the dictionary */
		BFile *theFile = new BFile(thePath.Path(),B_READ_ONLY);
		if (!theFile || theFile->InitCheck() != B_NO_ERROR) continue;
		BResources *theRes = new BResources(theFile,false);
		delete theFile;
		if (!theRes) continue;
		size_t len;
		nameArray[numDictionary] = (char*)(theRes->FindResource('CSTR',(int32)0,&len));
		if (nameArray[numDictionary])
		{
			totalSize += len + 1;
			++numDictionary;
		}

	/* Cleanup */
		delete theRes;
	}

/* Make a string large enough to hold all of the strings, plus comma delimiters */
	char *listString = (char*)(malloc(totalSize));
	listString[0] = (char)0;

/* Build the string */
	for (int32 i=0;i<numDictionary;++i)
	{
		strcat(listString,nameArray[i]);
		free(nameArray[i]);
		strcat(listString,",");
	}
	listString[totalSize] = (char)0;

/* Clean up */
	free(nameArray);
	*outList = listString;
	return numDictionary;
}

/**********************************************************************
*	Method:		language_name									*
*	Arguments:	int32	inLangID								*
*							the requested language ID			*
*				char*	inBuffer								*
*							optional buffer for the language name	*
*				int32	inBuffSize							*
*							number of bytes in inBuffer			*
*	Returns:		int32	String containing the language name of	*
*						inLangID								*
*	Description:	Gets the name of the language associated with		*
*				inLangID. If inBuffer is not null, it copies 		*
*				inBuffSize-1 bytes into inBuffer and returns it.	*
**********************************************************************/
char* language_name(int32 inLangID, char* inBuffer, size_t inBuffSize)
{
/* Try the local dictionary */
	size_t len;
	char *name = NULL;
	BResources *theRes = FindNextLocalDictionary(inLangID,0);
	if (theRes) name = (char*)(theRes->FindResource('CSTR',(int32)0,&len));

/* If nothing, try the common dictionary */
	if (!name)
	{
		delete theRes;
		theRes = FindCommonDictionary(inLangID);
		if (theRes) name = (char*)(theRes->FindResource('CSTR',(int32)0,&len));

	/* If nothing still, try the application */
		if (!name)
		{
			theRes = FindAppDictionary();
			if (theRes) name = (char*)(theRes->FindResource('CSTR',(int32)0,&len));
		}
	}

/* Copy as much of it as possible to inBuffer */
	if (name)
	{
		if (!inBuffer || !inBuffSize)
		{
			delete theRes;
			return name;
		}
		if (len > inBuffSize) len = inBuffSize;
		strncpy(inBuffer,name,len-1);
		inBuffer[len] = (char)0;
		free(name);
	}

/* Clean up */
	delete theRes;
	return inBuffer;
}

/**********************************************************************
*	Method:		language_id									*
*	Arguments:	char*	inBuffer								*
*							name of the requested language		*
*	Returns:		int32	Language ID associated with inBuffer		*
*	Description:	Gets the language ID of the language associated	*
*				with inBuffer.								*
**********************************************************************/
int32 language_id(char *inName)
{
/* Get the common dictionary directory */
	status_t err;
	BPath thePath;
	err = find_directory(B_COMMON_ETC_DIRECTORY,&thePath);
	if (err != B_NO_ERROR) return 0;
	thePath.Append("Blithe");
	BDirectory theDirectory(thePath.Path());
	theDirectory.Rewind();
	BEntry candidate;

/* Iterate over each dictionary until found */
	bool notFound = true;
	int32 id=0;
	while ((theDirectory.GetNextEntry(&candidate,true) == B_OK) && notFound)
	{
	/* Get the name of the dictionary */
		BFile *theFile = new BFile(thePath.Path(),B_READ_ONLY);
		if (!theFile || theFile->InitCheck() != B_NO_ERROR) continue;
		BResources *theRes = new BResources(theFile,false);
		if (!theRes) continue;
		delete theFile;
		size_t len;
		char *name = (char*)(theRes->FindResource('CSTR',(int32)0,&len));
		if (!name) continue;

	/* Compare it to the name passed in */
		if (strcmp(inName,name) == 0)
		{
			notFound = false;
			int32 *tmpID = (int32*)(theRes->FindResource('CSTR',(int32)0,&len));
			if (tmpID)
			{
				id = *tmpID;
				delete tmpID;
			}
		}
		free(name);

	/* Cleanup */
		delete theRes;
	}

	return id;
}

/**********************************************************************
*	Method:		get_raw_bits									*
*	Arguments:	<none>										*
*	Returns:		void*	The raw data for the Blithe bitmap		*
*	Description:	Gets the raw data to show the Blithe bitmap		*
**********************************************************************/
size_t get_raw_bits(void** theBits)
{
// Open the resource on this file
	BResources* theRes = FindSharedLib();

// Read in the raw data
	size_t theSize;
	void* bits = theRes->FindResource('bits',1,&theSize);

// Close the resource and return the data
	delete theRes;
	*theBits = bits;
	return theSize;
}

/**********************************************************************
*	Function:		FindSharedLib									*
*	Arguments:	<none>										*
*	Returns:		BResource*	the resource for the shared lib		*
*	Description:	Searches the BeOS, common, and user lib 			*
*				directories for the shared library and returns the	*
*				BResource*. If there is a problem, it returns null.	*
**********************************************************************/
BResources* FindSharedLib(void)
{
/* Look for the lib in the BeOS lib directory*/
	status_t err;
	BPath thePath;
	BResources* theRes=NULL;
	BFile *sharedLibFile=NULL;
	err = find_directory(B_BEOS_LIB_DIRECTORY,&thePath);
	if (err == B_NO_ERROR)
	{
		thePath.Append(BLITHE_LIB);
		sharedLibFile = new BFile(thePath.Path(),B_READ_ONLY);
		if (sharedLibFile && sharedLibFile->InitCheck() == B_NO_ERROR)
			theRes = new BResources(sharedLibFile,false);
		delete sharedLibFile;
		if (theRes) return theRes;
	}

/* Look for the lib in the Common lib directory */
	err = find_directory(B_COMMON_LIB_DIRECTORY,&thePath);
	if (err == B_NO_ERROR)
	{
		thePath.Append(BLITHE_LIB);
		BDirectory commonTest(thePath.Path());
		sharedLibFile = new BFile(thePath.Path(),B_READ_ONLY);
		if (sharedLibFile && sharedLibFile->InitCheck() == B_NO_ERROR)
			theRes = new BResources(sharedLibFile,false);
		delete sharedLibFile;
		if (theRes) return theRes;
	}

/* Look for the lib in the user lib directory */
	err = find_directory(B_USER_LIB_DIRECTORY,&thePath);
	if (err == B_NO_ERROR)
	{
		thePath.Append(BLITHE_LIB);
		BDirectory userTest(thePath.Path());
		sharedLibFile = new BFile(thePath.Path(),B_READ_ONLY);
		if (sharedLibFile && sharedLibFile->InitCheck() == B_NO_ERROR)
			theRes = new BResources(sharedLibFile,false);
		delete sharedLibFile;
	}
	return theRes;
}

/**********************************************************************
*	Method:		get_notify									*
*	Arguments:	<none>										*
*	Returns:		bool		True if we should notify running apps.	*
*	Description:	Reads the Blithe settings for the Notify flag.		*
**********************************************************************/
bool get_notify(void)
{
	bool *tmpNotify,notify=true;
	BPath settingsPath;
	size_t notifySize;
	status_t err;

/* Get the settings directory */
	err = find_directory(B_USER_SETTINGS_DIRECTORY,&settingsPath);
	if (err != B_NO_ERROR) return notify;
	settingsPath.Append("blithe_settings");

/* Open the settings file as a resource for read only */
	BFile *settingsFile = new BFile(settingsPath.Path(),B_READ_ONLY);
	if (!settingsFile || settingsFile->InitCheck() != B_NO_ERROR) return true;
	BResources *settingsRes = new BResources(settingsFile,false);
	delete settingsFile;
	if (!settingsRes) return true;

/* Get the notify resource */
	tmpNotify = (bool*)(settingsRes->FindResource('blnt',1,&notifySize));
	if (tmpNotify)
	{
		notify = *tmpNotify;
		free(tmpNotify);
	} else notify = 0;

/* Clean up*/
	delete settingsRes;
	return notify;
}

/**********************************************************************
*	Method:		FindNextLocalDictionary							*
*	Arguments:	int32	inLangID								*
*							the requested language ID			*
*				int32	i									*
*							index into the list of dictionaries	*
*	Returns:		BResources*	the BResource of the application		*
*				dictionary for language ID inLangID.				*
*	Description:	Looks for a file with the dictionary MIME type and	*
*				a language ID of inLangID. If found, returns the	*
*				BResource* for it. Otherwise returns NULL.		*
**********************************************************************/
BResources *FindNextLocalDictionary(int32 inLangID,int32 i)
{
	BPath thePath;
	app_info theAppInfo;
	status_t err = be_app->GetAppInfo(&theAppInfo);
	if (err != B_OK) return NULL;
	BEntry myEntry(&theAppInfo.ref,true);
	if (myEntry.InitCheck() != B_NO_ERROR) return NULL;
	BEntry parent;
	myEntry.GetParent(&parent);
	parent.GetPath(&thePath);
	thePath.Append("Blithe");
	return FindDictionaryInPath(thePath,inLangID,i);
}

/**********************************************************************
*	Method:		FindCommonDictionary							*
*	Arguments:	int32	inLangID								*
*							the requested language ID			*
*	Returns:		BResources*	the BResource of the common			*
*							dictionary for language ID inLangID.	*
*	Description:	Looks for a file with the dictionary MIME type and	*
*				a language ID of inLangID. If found, returns the	*
*				BResource* for it. Otherwise returns NULL.		*
**********************************************************************/
BResources *FindCommonDictionary(int32 inLangID)
{
	status_t err;
	BPath thePath;
	err = find_directory(B_COMMON_ETC_DIRECTORY,&thePath);
	if (err != B_NO_ERROR) return NULL;
	thePath.Append("Blithe");
	return FindDictionaryInPath(thePath,inLangID,0);
}

/**********************************************************************
*	Method:		FindDictionaryInPath							*
*	Arguments:	BPath	thePath								*
*							the path to search					*
*				int32	inLangID								*
*							the requested language ID			*
*	Returns:		BResources*	the BResource of the dictionary for	*
*							language ID inLangID.				*
*	Description:	Looks for a file with the dictionary MIME type and	*
*				a language ID of inLangID. If found, returns the	*
*				BResource* for it. Otherwise returns NULL.		*
**********************************************************************/
BResources *FindDictionaryInPath(BPath thePath,int32 inLangID,int32 i)
{
/* Take care of the default case */
	if (inLangID == BL_DEFAULT_LANGUAGE) inLangID = get_default_language();

// Get ready
	BDirectory theDirectory(thePath.Path());
	theDirectory.Rewind();
	BEntry candidate;
	int32 j=0;
	while (j<i && theDirectory.GetNextEntry(&candidate,true) == B_OK) ++j;

// Check each dictionary for the language id
	bool found=false;
	while (theDirectory.GetNextEntry(&candidate,true) == B_OK)
	{
		BNode theNode(&candidate);
		char mimeType[1024];
		BNodeInfo(&theNode).GetType(mimeType);
		if (strncmp(mimeType,"application/x-vnd.Bastion-BlitheDict",36) != 0) continue;

		BFile theFile(&candidate,B_READ_ONLY);
		if (theFile.InitCheck() != B_NO_ERROR) continue;
		BResources *theResource = new BResources(&theFile);
		if (!theResource) continue;
		size_t len;
		int32 *id;
		id = (int32 *)(theResource->FindResource('LONG',1,&len));
		if (!id) continue;
		found = (inLangID == *id);
		free(id);
		if (found) return theResource; 
		delete theResource;
	}
	return NULL;
}

/**********************************************************************
*	Method:		GetFirstDictionary								*
*	Arguments:	<none>										*
*	Returns:		int32	The first language ID it comes across in	*
*						the common dictionary directory.			*
*	Description:	Searches the common dictionaries for a language ID.	*
*				This is used in case we don't have a valid		*
*				language ID and we need one.					*
**********************************************************************/
int32 GetFirstDictionary(void)
{
// Get the path
	status_t err;
	BPath thePath;
	err = find_directory(B_COMMON_ETC_DIRECTORY,&thePath);
	if (err != B_NO_ERROR) return 0;
	thePath.Append("Blithe");
	BDirectory theDirectory(thePath.Path());
	theDirectory.Rewind();
	BEntry candidate;

// Get the first valid dictionary
	while (theDirectory.GetNextEntry(&candidate,true) == B_OK)
	{
		BNode theNode(&candidate);
		char mimeType[1024];
		BNodeInfo(&theNode).GetType(mimeType);
		if (strncmp(mimeType,"application/x-vnd.Bastion-BlitheDict",36) != 0) continue;

		BFile theFile(&candidate,B_READ_ONLY);
		if (theFile.InitCheck() != B_NO_ERROR) continue;
		BResources *theResource = new BResources(&theFile);
		if (!theResource) continue;
		size_t len;
		int32 *id;
		id = (int32 *)(theResource->FindResource('LONG',1,&len));
		if (!id) continue;

		int32 val = *id;
		free(id);
		return val;
	}
	return BL_ENGLISH_LANGUAGE;
}

/**********************************************************************
*	Method:		FindAppDictionary								*
*	Arguments:	<none>										*
*	Returns:		BResources*	The resource of the applicaton.		*
*	Description:	Gets the resource of the application so that we	*
*				can look inside the application. Although the		*
*				application is among the images, in most cases, the	*
*				resource is probably inside the application itself	*
*				and checking all of the images can be a time		*
*				consuming process.								*
**********************************************************************/
BResources *FindAppDictionary(void)
{
	BFile file; 
	app_info info; 

	be_app->GetAppInfo(&info);
	file.SetTo(&info.ref, B_READ_ONLY);
	return new BResources(&file);
}

/**********************************************************************
*	Method:		GetHex										*
*	Arguments:	int32	inStringID							*
*							the requested entry ID				*
*	Returns:		char*	the hex of inStringID as a string		*
*	Description:	Converts inStringID to a hex stored as a string.	*
*				This is used in case we can't find any entry		*
*				associated with inStringID in any dictionary.		*
**********************************************************************/
char *GetHex(int32 inStringID)
{
	char *theString = (char*)(malloc(11));
	sprintf(theString,"%lx",inStringID);
	return theString;
}

/**********************************************************************
*	Function:		ConvertEscapes								*
*	Arguments:	char*	entry								*
*							the entry string					*
*	Returns:		<none>										*
*	Description:	Converts escape codes to their proper form.		*
**********************************************************************/
void ConvertEscapes(char* entry)
{
// Iterate over each character
	char* t=entry;
	char* temp = (char*)(malloc(strlen(entry)+1));
	while (t && t[0])
	{
	// Is t a '\'
		if (t[0] == '\\')
		{
		// Convert it to the proper code based on t[1]
			switch (t[1])
			{
			case 'n':
			// New line (in BeOS linefeed)
				t[0] = B_ENTER;
				break;
			case 'b':
			// Backspace
				t[0] = B_BACKSPACE;
				break;
			case 'f':
			// Formfeed
				t[0] = B_PAGE_DOWN;
				break;
			case 'r':
			// Carriage return
				t[0] = 13;
				break;
			case 't':
			// Horizontal tab
				t[0] = B_TAB;
				break;
			case '\"':
			// Double quote
				--t;
				break;
			case '\'':
			// Single quote
				--t;
				break;
			case '0':
			// Null
				t[0] = 0;
				break;
			case '\\':
			// Backslash
				--t;
				break;
			}

		// Move the string over
			strcpy(temp,t+2);
			strcpy(t+1,temp);
		}
		++t;
	}
}